function [A_est time iter_cnt obj_track]=WAVMAX(X,N,const,show_flag,index)

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Worst-case Alternating Volume Maximization (WAVMAX) Algorithm %
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ===Input=========
% X: hyperspectral data - M by L matrix (M: number of spectral bands, L: number of pixels)
% N: number of endmembers
% const: error tolerance (back-off distance)
% show_flag: 0-don't display, 1-display
% index: pure pixel indices estimated by any endmember extraction method 
% ===Output========
% A_est: robust endmember estimates
% time: computation time 
% iter_cnt: number of iterations (only outter loop)
% obj_track: historical objective values in all the iterations 
% =================
% Programmer: Dr. Tsung-Han Chan, Email: thchan@ieee.org 
% The WAVMAX algorithm is proposed in the following paper: 
% % T.-H. Chan, W.-K. Ma, A. Ambikapathi, and C.-Y. Chi, A simplex volume 
% % maximization framework for hyperspectral endmember extraction, to appear 
% % in IEEE Trans. on Geoscience and Remote Sensing  Special Issue on Spectral 
% % Unmixing of Remotely Sensed Data, 2011
% ================= 
% Table IV 

t0 = clock;
%----------- Define default parameters------------------
TOL_obj = 5e-5; % convergence tolerance in the alternating optimization for prob. (35)
tol_U = 3e-4; % accuracy tolerence in the alternating optimization for prob. (41)
iter_k = 5; % maximum number of subgradient iterations
alpha = 1; % subgradient step size

%------ Dimension reduction using affine set fitting-----------------
[M,L] = size(X); 
d = mean(X,2); % eq. (4) - mean vector of the data cloud
U = X-d*ones(1,L); % mean-removed data cloud 
OPTS.disp = 0;
[C D] = eigs(U*U',N-1,'LM',OPTS); % eq. (5) - the first N-1 principal eigenvectors
Xd = C'*U;  % eq. (6) - Dimension reduced data vectors (Xd is (N-1)-by-L)

% ------ Step 1. in Table IV -------------
E= [Xd(:,index);ones(1,N)]; % initial corner points
THE = zeros(L,N); 
for i=1:N
    THE(index(i),i) = 1; % initial theta
end
U = sub_e(Xd,THE,tol_U, const); % initial U obtained by Table III

% ------ Step 2. in Table IV 
E(1:N-1,:) = Xd*THE - U; % initial E
obj0 = abs(det(E))/factorial(N-1);

% ------ Step 3. in Table IV
j = 0; rec = 1; iter_cnt = 0;
while (rec > TOL_obj) & (iter_cnt<2*N) % outter loop (Find THE alternatingly)
    j = j+1;
    if j>N; j=1; end 
    k = 0; 
    alpha_k = alpha;
    while k < iter_k      % medium loop (find the optimal THE(:,j) with the given optimal U)
        % ------ Step 3.1 in Table IV 
        k = k+1;
        % ------ Step 3.2 in Table IV 
        b=[];
        for i=1:N;
            Eij=[E(1:i-1,1:j-1),E(1:i-1,j+1:N);E(i+1:N,1:j-1),E(i+1:N,j+1:N)];
            b=[b;(-1)^(i+j)*det(Eij)];  
        end
        b1 = b(1:N-1);  % calculate b_j (based on E)   
        alpha_k = alpha_k/sqrt(k);
        U_old = U;
        THE_old = THE; % save the optimal THE(:,j) before updation
        % ------ Step 3.3 in Table IV 
        THE(:,j) = THE(:,j) + alpha_k*Xd'*b1;
        while (any(abs(sum(THE(:,j),1)-1)>1e-5)||(any(any(THE(:,j)<0)))) % projection of THE(:,j) onto a simplex
            THE(:,j) = THE(:,j)-ones(L,1)*(sum(THE(:,j),1)-1)/L;
            THE(THE(:,j)<0,j) = 0;
        end
        % ------ Step 3.4 in Table IV 
        U = sub_e(Xd,THE,tol_U, const);
        
        % ------ Step 3.5 in Table IV 
        E(1:N-1,:) = Xd*THE - U;
        obj_new = det(E);
        obj_old = det([Xd*THE_old - U_old; ones(1,N)]);
        if obj_new < obj_old; 
           THE = THE_old;  U = U_old;
           E(1:N-1,:) = Xd*THE - U;
        end  
        obj_track(iter_cnt+1,j,k) = det(E);
    end    
           
    %------Step 4. and Step 5. in Table IV---------------
    if j == N
        iter_cnt = iter_cnt+1;
        rec = abs(obj0-abs(det(E))/factorial(N-1))/obj0;
        obj0 = abs(det(E))/factorial(N-1);
        if show_flag
            disp(' ');
            disp(strcat('Number of iterations: ', num2str(iter_cnt)))
            disp(strcat('Relative change in objective: ', num2str(rec)))
            disp(strcat('Current volume: ', num2str(obj0)))
        end
    end
end
%--------Affine mapping eq. (12)------------------
A_est = C*E(1:N-1,:)+d*ones(1,N);
%------------------------------------------
time = etime(clock,t0);



function [U] = sub_e(Xd,THE,tol_U,const)

% ===Input=========
% Xd: dimension reduced data - (N-1) by L matrix 
% THE: the given theta_1,...,theta_N
% tol_U: convergence tolerance 
% const: error tolerance (back-off distance) 
% ===Output========
% U: estimated back-off vectors
% =================
% Table III 

l=0; 
er_U = 10;      
n = size(Xd,1); 
N = n+1;
% --- Step 1. in Table III 
U = zeros(N-1,N);
E = ones(N,N);
E(1:N-1,:) = Xd*THE - U; 
count = 0;
while (er_U > tol_U) & (count<50) % inner loop (find the optimal U with the given THE)
      % --- Step 2. in Table III 
      count = count+1;
      l = l+1;
      if l>N; l=1; end
      U_old = U; 
      b = [];
      for i=1:N
           Eil = [E(1:i-1,1:l-1),E(1:i-1,l+1:N);E(i+1:N,1:l-1),E(i+1:N,l+1:N)];
           b = [b;(-1)^(i+l)*det(Eil)];  
      end
      b = b(1:N-1);
      b([b~=0]) = b([b~=0])./(max(abs(b([b~=0])))*ones(length(find(b~=0)),1)); 
      temp = Xd*THE(:,l);
      % --- Step 3. in Table III 
      U(:,l) = const*b'./norm(b); 
      % --- Step 4. and 5. in Table III 
      E(1:N-1,l) = temp - U(:,l); % update E with the given THE and U until convergence
      if l==N
         er_U = norm(U-U_old);
      end
end